import java.io.*;
import java.util.*;

class DatabaseManager {
   private IOManager iomgr;
   public static final String ElementsDB = "Elements.dat";
   public static final String EnthalpyDB = "Enthalpy.dat";
   public static final String LayoutDB = "Layout.dat";
   public static final String ClassesDB = "Class.dat";
   public static final String StatesDB = "State.dat";
   public static final String TypesDB = "Types.dat";

   private Map elements;
   private List layout;
   private Map classes;
   private Map states;
   private Map types;

   public final Map getElements() {return elements;}
   public final List getLayout() {return layout;}
   public final Map getClasses() {return classes;}
   public final Map getStates() {return states;}
   public final Map getTypes() {return types;}

   private LoadStatusManager lsmgr;
   private ConfigManager cfgmgr;

   public TElement getElement(int number) {
       return (TElement)elements.get(new Integer(number));
   }

   public DatabaseManager(LoadStatusManager lsmgr, IOManager iom, ConfigManager c) {
       iomgr = iom;
       this.lsmgr = lsmgr;
       cfgmgr = c;

       //Now read elements database
       try {
           lsmgr.setProgress("Reading atomic database...");
           elements = getElements(iomgr.getInputStream(ElementsDB), iomgr.getInputStream(EnthalpyDB));

           lsmgr.setProgress("Reading table layout...");
           layout = getLayout(iomgr.getInputStream(LayoutDB));

           lsmgr.setProgress("Reading classes...");
           classes = getClasses(iomgr.getInputStream(ClassesDB));

           lsmgr.setProgress("Reading states...");
           states = getStates(iomgr.getInputStream(StatesDB));

           lsmgr.setProgress("Reading element types...");
           types = getTypes(iomgr.getInputStream(TypesDB));
       } catch (Exception e) {
           e.printStackTrace();
       }
   }

   private Map getElements(InputStream elements, InputStream enthalpies) throws Exception {
       BufferedReader d = new BufferedReader(new InputStreamReader(elements));
       BufferedReader d2 = new BufferedReader(new InputStreamReader(enthalpies));

       Map h = new HashMap();

       String s;
       TElement t;

       s = d.readLine(); //eat the headers...
       StringTokenizer z = new StringTokenizer(s, "\t", false);
       int iNumTokens = z.countTokens();

       d2.readLine(); //eat enthalpy headers...
       int count = 0;

       while( (s = d.readLine()) != null) {
           count++;
           if((count % 10) == 0) lsmgr.setProgress("Reading element "+count+"...");
           t = TElement.decode(s, iNumTokens);
           t.enthalpies = decodeEnthalpies(d2.readLine()); //enthalpies MUST have the same sequence as elements!
           h.put(new Integer(t.AtomicNumber), t);
       }

       elements.close();
       enthalpies.close();
       return h;
   }

   private List getLayout(InputStream inFile) throws Exception {
       BufferedReader d = new BufferedReader(new InputStreamReader(inFile));

       List l = new ArrayList(120);

       String s;
       TElement t;

       s = d.readLine(); //eat the headers...
       StringTokenizer z = new StringTokenizer(s, "\t", false);
       l.add( z.nextToken() );
       l.add( z.nextToken() );

       while( (s = d.readLine()) != null) {
           z = new StringTokenizer(s, "\t", false);

           while(z.hasMoreTokens())
               l.add(z.nextToken());
       }
       inFile.close();
       return l;
   }

   private float[] decodeEnthalpies(String input) throws Exception{
       StringTokenizer z = new StringTokenizer(input, "\t", false);

       z.nextToken(); //skip atomic number
       int c = z.countTokens();

       float enthalpies[] = new float[c];

       for(int i = 0; i < c; i++)
           enthalpies[i] = Float.valueOf(z.nextToken()).floatValue();

       return enthalpies;
   }

   private Map getClasses(InputStream inFile) throws Exception {
       BufferedReader d = new BufferedReader(new InputStreamReader(inFile));
       Map m = new HashMap();
       String s;

       d.readLine(); //eat header

       while( (s = d.readLine()) != null) {
           TClass c = TClass.decode(s);
           m.put(new Integer(c.number), c);
       }
       inFile.close();
       return m;
   }

   private Map getStates(InputStream inFile) throws Exception {
       BufferedReader d = new BufferedReader(new InputStreamReader(inFile));
       Map m = new HashMap();
       String s;

       d.readLine(); //eat header

       while( (s = d.readLine()) != null) {
           TState st = TState.decode(s);
           m.put(new Integer(st.number), st);
       }
       inFile.close();
       return m;
   }

   private Map getTypes(InputStream inFile) throws Exception {
       BufferedReader d = new BufferedReader(new InputStreamReader(inFile));
       Map m = new HashMap();
       String s;

       d.readLine(); //eat header

       while( (s = d.readLine()) != null) {
           TElementType et = TElementType.decode(s);
           m.put(new Integer(et.number), et);
       }
       inFile.close();
       return m;
   }

   public void saveEverything(DataOutputStream os) throws IOException {
       Iterator i = elements.values().iterator();

       while(i.hasNext()) {
           os.writeBytes(AllElementInfo.getAtomInfoAsText((TElement)i.next(), this));
           os.writeBytes(cfgmgr.getCrlf());
           os.writeBytes(cfgmgr.getCrlf());
           os.writeBytes("======================================");
           os.writeBytes(cfgmgr.getCrlf());
           os.writeBytes(cfgmgr.getCrlf());
       }
   }
}
