/**
 * Plot the Mandelbrot set with red-coloring
 */
package grid.client.fractal;

import grid.Workable;
import grid.worker.TaskWorker;
import java.awt.image.BufferedImage;

/**
 * Plot the Mandelbrot set, with red layering.
 * The algorithm used is z(n+1) = z(n)^2 + c;
 */
public class MandelPlot implements Workable {
	private int img_x, img_y, img_w, img_h, chunk_x, chunk_y;
	private double min_x, max_x, min_y, max_y, c;

	private static final double THRESHOLD = 16.0;
	private static final int LOOPS = 1000;

	public MandelPlot(int img_x, int img_y, int img_w, int img_h, int chunk_x,
		int chunk_y, double min_x, double max_x, double min_y, double max_y,
		double c
	) {
		this.img_x = img_x;
		this.img_y = img_y;
		this.img_w = img_w;
		this.img_h = img_h;
		this.chunk_x = chunk_x;
		this.chunk_y = chunk_y;
		this.min_x = min_x;
		this.max_x = max_x;
		this.min_y = min_y;
		this.max_y = max_y;
		this.c = c;
	}

	public Object execute(TaskWorker controller) {
		BufferedImage bi = new BufferedImage(chunk_x, chunk_y,
			BufferedImage.TYPE_INT_ARGB);
		int pixels[][] = new int[chunk_y][chunk_x];

		//for each pixel in the chunk...
		for(int y = 0; y < chunk_y; y++) {
			for(int x = 0 ; x < chunk_x; x++) {
				pixels[y][x] = getColor(x, y);
			}
		}

		return (Object) pixels;
	}

	private int getColor(int x, int y) {
		double alpha = ((((double) x + img_x) / img_w) * (max_x - min_x)) + min_x;
		double beta  = ((((double) y + img_y) / img_h) * (max_y - min_y)) + min_y;

		double a = alpha;
		double b = beta;
		double new_a, new_b;
		double sq = (a * a) + (b * b);

		// Expand the sum series 100 terms...
		int i;
		for(i = 0; i < LOOPS && sq < THRESHOLD; i++) {
			
			new_a = (a * a) - (b * b) + alpha;
			new_b = (2 * a * b) + beta;
			//*/
			/*
			new_a = sq - 0.75;
			new_b = 2 * a * b;
			//*/
			a = new_a;
			b = new_b;
			sq = (a * a) + (b * b);
		}

		// If the sum is still in the allowed range,
		// then return black.
		if(sq < THRESHOLD) {
			return 0xFF000000;
		}

		// Otherwise, figure out what color we get...
		/*
		int redness = (int)Math.round( ((LOOPS - (double)i) / LOOPS) * 255 );
		return 0xFF000000 | (redness << 16);
		*/
		int colorness = (int)Math.round( ((LOOPS - (double)i) / LOOPS) * 0x00FFFFFF );
		return 0xFF000000 | colorness;
	}

	public int getX() {return img_x;}
	public int getY() {return img_y;}
	public int getWidth() {return chunk_x;}
	public int getHeight() {return chunk_y;}
}
