package grid.manager;

import grid.worker.Task;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.io.Serializable;
import java.util.Iterator;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

public class Manager extends UnicastRemoteObject implements WorkerManager,
	ClientManager, Serializable
{
	private static long taskIdCounter = 0;
	private List jobs, waiting, processing;
	private Map jobsByJob, jobsByTask;
	private JobCleaner jc;
	private List scheduleListeners, finishListeners, statusChangeListeners;

	public Manager() throws RemoteException {
		super();
		processing = new LinkedList();
		waiting = new LinkedList();
		jobs = new LinkedList();
		jobsByJob = new HashMap();
		jobsByTask = new HashMap();
		scheduleListeners = new LinkedList();
		finishListeners = new LinkedList();
		statusChangeListeners = new LinkedList();

		jc = new JobCleaner(this);
		jc.start();
	}

	public void scheduleJob(Job job) {
		if(job.finished == null) return;
		
		//create a new job tuple
		JobTuple jt = new JobTuple(
			job,
			new Task(job.work, job.jobID+"-"+(taskIdCounter++))
		);

		//put it in the list
		synchronized (jobs) {
			jobs.add(jt);
			jobsByJob.put(jt.job.jobID, jt);
			jobsByTask.put(jt.task.taskID, jt);
			waiting.add(jt);
		}

		fireScheduleEvent(new int[] {jobs.size()-1});
	}

	public void scheduleJobs(Job jobList[]) {
		int x = jobList.length;
		int indices[] = new int[x];

		for(int i = 0; i < x; i++) {
			if(jobList[i].finished == null) return;
			//make new job
			JobTuple jt = new JobTuple(
				jobList[i],
				new Task(jobList[i].work, jobList[i].jobID+"-"+(taskIdCounter++))
			);

			indices[i] = x + i;

			//add to list
			synchronized (jobs) {
				jobs.add(jt);
				jobsByJob.put(jt.job.jobID, jt);
				jobsByTask.put(jt.task.taskID, jt);
				waiting.add(jt);
			}
		}

		fireScheduleEvent(indices);
	}

	public void abortJob(String jobID) {
		int idx;
		
		synchronized (jobs) {
			JobTuple jt = (JobTuple)jobsByJob.get(jobID);
			if(jt == null) return;

			idx = jobs.indexOf(jt);

			abortJobTuple(jt);
		}
		
		fireFinishedEvent(new int[]{idx});
	}

	public void abortJobs(String jobIDs[]) {
		int x = jobIDs.length;
		int indices[] = new int[x];
		int idx;

		synchronized (jobs) {
			for(int i = 0; i < x; i++) {
				JobTuple jt = (JobTuple)jobsByJob.get(jobIDs[i]);
				if(jt == null) return;

				idx = jobs.indexOf(jt);
				indices[i] = idx;

				abortJobTuple(jt);
			}
		}

		fireFinishedEvent(indices);
	}

	private void abortJobTuple(JobTuple jt) {
		synchronized(jobs) {
			jobsByJob.remove(jt.job.jobID);
			jobsByTask.remove(jt.task.taskID);
			jobs.remove(jt);
			waiting.remove(jt);
			processing.remove(jt);
		}
	}

	public Task findTask(float goodness, String workerID) {
		JobTuple jt = null;

		synchronized (jobs) {
			if(waiting.isEmpty()) return null;

			jt = (JobTuple)waiting.remove(0);
		}

		System.out.println("Dispatched job "+jt.job.jobID
			+" to worker "+workerID+" as task "
			+jt.task.taskID+".");
		int oldstatus = jt.status;
		jt.status = JobTuple.PROCESSING;
		jt.processingExpiration = System.currentTimeMillis()
			+ jt.job.processingExpiration;
		jt.workerID = workerID;

		processing.add(jt);

		fireStatusChangeEvent(jt.job.jobID, oldstatus, jt.status);
		return jt.task;
	}

	public void abortTask(String taskID) {
		JobTuple jt;
		synchronized (jobs) {
			jt = (JobTuple)jobsByTask.get(taskID);
			if(jt == null) return;
			if(processing.indexOf(jt) == -1) return;

			processing.remove(jt);
			waiting.add(1, jt);
		}
		System.out.println("Task "+taskID+" was aborted.");
		int oldstatus = jt.status;
		jt.status = JobTuple.PENDING;
		
		fireStatusChangeEvent(jt.job.jobID, oldstatus, jt.status);
	}

	public void finishTask(String taskID, Object result) {
		JobTuple jt;
		int idx;
		
		synchronized (jobs) {
			jt = (JobTuple)jobsByTask.get(taskID);
			if(jt == null) return;

			idx = jobs.indexOf(jt);

			abortJobTuple(jt);
		}
		System.out.println("Finished task "+taskID+".");

		try {
			jt.job.finished.jobFinished(jt.job.jobID, result);
		} catch (Exception e) {
			if(e instanceof java.rmi.ConnectException) return;
			e.printStackTrace();
		}

		fireFinishedEvent(new int[] {idx});
	}

	protected void expireTasks() {
		long now = System.currentTimeMillis();
			
		synchronized (jobs) {
			Iterator ji;
			ji = jobs.iterator();

			while(ji.hasNext()) {
				JobTuple jt = (JobTuple)ji.next();

				if(jt.status != JobTuple.PROCESSING) continue;

				if(jt.processingExpiration <= now) {
					System.out.println("Task "
						+jt.task.taskID+" expired.");
					processing.remove(jt);
					waiting.add(jt);
					int oldstatus = jt.status;
					jt.status = JobTuple.PENDING;

					fireStatusChangeEvent(jt.job.jobID, oldstatus, jt.status);
				}
			}
		}
	}

	public int getJobCount() {
		return jobs.size();
	}

	public int getJobNumber(String jobID) {
		Object o = jobsByJob.get(jobID);
		return jobs.indexOf(o);
	}
	
	public String getJobID(int job) {
		if(job >= jobs.size() || job < 0) return null;
		
		JobTuple jt = (JobTuple)jobs.get(job);
		return jt.job.jobID;
	}

	public int getJobStatus(int job) {
		if(job >= jobs.size() || job < 0) return -1;
		
		JobTuple jt = (JobTuple)jobs.get(job);
		return jt.status;
	}

	public String getJobWorkerID(int job) {
		if(job >= jobs.size() || job < 0) return null;
		
		JobTuple jt = (JobTuple)jobs.get(job);
		if(jt.status == JobTuple.PROCESSING) {
			return jt.workerID;
		} else {
			return "<none>";
		}
	}

	public void fireScheduleEvent(int serials[]) {
		JobScheduleEvent jse = new JobScheduleEvent(this, serials);

		Iterator i = scheduleListeners.iterator();
		while(i.hasNext()) {
			JobScheduleListener jsl = (JobScheduleListener)i.next();
			jsl.jobScheduled(jse);
		}
	}
	public void addScheduleListener(JobScheduleListener jsl) {
		scheduleListeners.add(jsl);
	}
	public void removeScheduleListener(JobScheduleListener jsl) {
		scheduleListeners.remove(jsl);
	}

	public void fireFinishedEvent(int serials[]) {
		JobFinishEvent jfe = new JobFinishEvent(this, serials);

		Iterator i = finishListeners.iterator();
		while(i.hasNext()) {
			JobFinishListener jfl = (JobFinishListener)i.next();
			jfl.jobFinished(jfe);
		}
	}
	public void addFinishListener(JobFinishListener jfl) {
		finishListeners.add(jfl);
	}
	public void removeFinishListener(JobFinishListener jfl) {
		finishListeners.remove(jfl);
	}

	public void fireStatusChangeEvent(String jobID, int oldS, int newS) {
		JobStatusChangeEvent jsce =
			new JobStatusChangeEvent(this, jobID, oldS, newS);

		Iterator i = statusChangeListeners.iterator();
		while(i.hasNext()) {
			JobStatusChangeListener jscl =
				(JobStatusChangeListener)i.next();
			jscl.jobStatusChanged(jsce);
		}
	}
	public void addStatusChangeListener(JobStatusChangeListener jscl) {
		statusChangeListeners.add(jscl);
	}
	public void removeStatusChangeListener(JobStatusChangeListener jscl) {
		statusChangeListeners.remove(jscl);
	}

}

class JobCleaner extends Thread {
	private Manager man;

	public JobCleaner(Manager man) {
		this.man = man;
	}

	public void run() {
		while(true) {
			try {
				Thread.sleep(30000);
			} catch (Exception e) {}
			man.expireTasks();
		}
	}
}


class JobTuple {
	public static final int PENDING = 0;
	public static final int PROCESSING = 1;
	public static final int FINISHED = 2;

	public Job job;
	public Task task;
	public int status;
	public Object result;
	public long processingExpiration;
	public String workerID;

	public JobTuple(Job job, Task task) {
		this.job = job;
		this.task = task;
		this.status = JobTuple.PENDING;
		this.result = null;
	}
}
