package grid.worker;

import grid.util.Defaults;
import grid.manager.WorkerManager;
import grid.worker.TaskWorker;
import grid.worker.StatusChangeListener;
import java.rmi.Naming;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.net.InetAddress;

/**
 * This is the basic worker representation.
 * It exists to contact a Manager, ask it for work, do the work, and
 * finally return a result.
 */
public class Worker extends Thread implements TaskWorker {
	private String registry;
	private Task task;
	private int status;
	private WorkerManager wm;
	private String progress;
	private boolean proceed;
	private Object restartWaiter;

	public Worker(String registry) {
		this.registry = registry;
		this.status = DISCONNECTED;
		statusChangeListeners = new LinkedList();
		progress = null;
		proceed = false;
		restartWaiter = new Object();
	}

	public String getRegistry() {
		return registry;
	}

	public int getStatus() {
		return status;
	}

	public String getTaskID() {
		if(task == null) return "<none>";

		return task.taskID;
	}

	public String getTaskProgress() {
		return progress;
	}

	public static final int CONNECTING = 0;
	public static final int DISCONNECTED = 1;
	public static final int WAITING = 2;
	public static final int RUNNING = 3;

	public void connect() {
		if(status != DISCONNECTED) return;
		setStatus(CONNECTING);

		String name = "//" + registry + "/" + Defaults.managerService;
		wm = null;
		try {
			wm = (WorkerManager)Naming.lookup(name);
		} catch (Exception e) {
			System.err.println("Unable to find manager at "+registry+"!  Aborting.");
			System.err.println(e.getMessage());
			setStatus(DISCONNECTED);
			return;
		}
		setStatus(WAITING);
	}

	public void reconnect() {
		synchronized(restartWaiter) {
			restartWaiter.notify();
		}
	}

	public void run() {
		while(true) {
			performWork();
			synchronized(restartWaiter) {
				try {
					restartWaiter.wait();
				} catch (Exception e){}
			}
		}
	}

	public void performWork() {
		//try to connect, if we're not already connected.
		if(status == DISCONNECTED) {
			connect();
			if(status == DISCONNECTED) {
				return;
			}
		}

		//now try to figure out what our hostname is.
		InetAddress localhost = null;
		String workerID;
		try {
			localhost = InetAddress.getLocalHost();
			workerID = localhost.getHostName();
		} catch (Exception e) {
			//??? no IP addr?  Hrm...
			workerID = ""+hashCode();
		}

		//go into the main execution loop.
		while(true) {
			try {
				//calculate a goodness measure.
				//Ideally, this would be based on how fast the system
				//is and how busy it is, but for now we just say 1.
				float goodness = 1.0f;
				
				//try to find us a manager
				if(isInterrupted()) return;
				task = wm.findTask(goodness, workerID);
				while(task == null) {
					try {
						Thread.sleep(1000);
					} catch (Exception d) {}
					if(isInterrupted()) return;
					task = wm.findTask(goodness, workerID);
				}

				setStatus(RUNNING);

				System.out.println("Obtained task "+task.taskID+".");
			} catch (Exception e) {
				//if we get here, then something bad happened to the
				//server. abort.
				setStatus(DISCONNECTED);
				if(e instanceof java.rmi.ConnectException) {
					System.err.println("Server disconnected me.  Bye.");
					return;
				} else if(e instanceof java.rmi.UnmarshalException) {
					System.err.println("Connection to server failed.  Bye.");
					return;
				}
				e.printStackTrace();
				return;
			}

			if(isInterrupted()) return;
			Object o = null;
			proceed = true;
			try {
				//now execute the task
				o = task.work.execute(this);
			} catch (Exception e) {
				//if task fails, print stack trace, abort, and
				//go find another task.
				e.printStackTrace();
				try {
					wm.abortTask(task.taskID);
				} catch (Exception x) {
					setStatus(DISCONNECTED);
					return;
				}

				setStatus(WAITING);
				if(isInterrupted()) return;
				continue;
			}

			System.out.println("Completed task "+task.taskID+".");

			if(isInterrupted()) return;
			try {
				//signal that we're done.
				if(proceed) {
					wm.finishTask(task.taskID, o);
				}
			} catch (Exception e) {
				//something bad happened here...
				e.printStackTrace();
				setStatus(DISCONNECTED);
				return;
			}
			proceed = false;

			task = null;
			setStatus(WAITING);
			if(isInterrupted()) return;
		}
	}

	private List statusChangeListeners;

	private void setStatus(int newstatus) {
		StatusChangeEvent sce = new StatusChangeEvent(this, status, newstatus);
		status = newstatus;

		Iterator i = statusChangeListeners.iterator();

		while(i.hasNext()) {
			((StatusChangeListener)i.next()).statusChanged(sce);
		}
	}

	public void addStatusChangeListener(StatusChangeListener scl) {
		statusChangeListeners.add(scl);
	}

	public void removeStatusChangeListener(StatusChangeListener scl) {
		statusChangeListeners.remove(scl);
	}

	public boolean shouldProceed() {
		return proceed;
	}

	public void interrupt() {
		abortTask();
		super.interrupt();
	}

	public void abortTask() {
		//determine if task is aborted.
		//if it isnt, then send the abort to the manager
		if(proceed == false) return;

		proceed = false;
		try {
			wm.abortTask(task.taskID);
		} catch (Exception e) {
			e.printStackTrace();
			setStatus(DISCONNECTED);
			return;
		}
	}
}
				
