package grid.worker;

import java.awt.Window;
import java.awt.Frame;
import java.awt.Insets;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JButton;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.ListSelectionEvent;

public class WorkerController extends JPanel implements WindowListener {
	private JTable workerTable;
	private JButton addButton, stopButton, killButton, exitButton, reconnectButton;

	private List workerList;
	private WorkerTableModel wtm;
	private StatusChangeHandler sch;
	private Frame frame;

	public WorkerController(Frame frame) {
		this.frame = frame;
		init();
	}

	public WorkerController() {
		init();
	}

	private void init() {
		//set up data objects
		workerList = new ArrayList();

		//set up layouts
		GridBagLayout gb = new GridBagLayout();
		setLayout(gb);
		GridBagConstraints c = new GridBagConstraints();

		//set up components
		wtm = new WorkerTableModel(workerList);
		workerTable = new JTable(wtm);

		addButton = new JButton("Add");
		addButton.setMnemonic('A');

		stopButton = new JButton("Stop");
		stopButton.setMnemonic('S');
		stopButton.setEnabled(false);

		killButton = new JButton("Kill");
		killButton.setMnemonic('k');
		killButton.setEnabled(false);

		exitButton = new JButton("Exit");
		exitButton.setMnemonic('x');

		reconnectButton = new JButton("Reconnect");
		reconnectButton.setMnemonic('R');
		reconnectButton.setEnabled(false);

		//now insert them into panel
		c.insets = new Insets(10, 10, 10, 10);
		c.fill = GridBagConstraints.HORIZONTAL;
		c.weightx = 1.0;

		JLabel label = new JLabel("Worker threads:");
		c.gridx = 0;
		c.gridy = 0;
		c.gridwidth = 4;
		gb.setConstraints(label, c);
		add(label);

		c.gridwidth = 1;
		c.gridx = 0;
		c.gridy = 2;
		gb.setConstraints(addButton, c);
		add(addButton);

		c.gridx = 1;
		gb.setConstraints(stopButton, c);
		add(stopButton);

		c.gridx = 2;
		gb.setConstraints(killButton, c);
		add(killButton);

		c.gridx = 3;
		gb.setConstraints(reconnectButton, c);
		add(reconnectButton);

		if(frame != null) {
			c.gridx = 4;
			gb.setConstraints(exitButton, c);
			add(exitButton);
		}

		c.gridx = 0;
		c.gridy = 1;
		c.gridwidth = 5;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.BOTH;
		JScrollPane jsp = new JScrollPane(workerTable);
		gb.setConstraints(jsp, c);
		add(jsp);

		//set up event handlers
		sch = new StatusChangeHandler();
		addButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					AddWorkerDialog awd = new AddWorkerDialog(frame);
					List w = awd.getWorker();
					if(w == null) return;
					int x = workerList.size();
					workerList.addAll(w);
					wtm.fireTableRowsInserted(x-1, x-1+w.size());

					Iterator i = w.iterator();
					while(i.hasNext()) {
						Worker q = (Worker)i.next();
						q.addStatusChangeListener(sch);
					}
				}
			}
		);

		if(frame != null) {
			exitButton.addActionListener(
				new ExitButtonHandler(frame)
			);
		}

		reconnectButton.addActionListener(new ReconnectHandler());
		stopButton.addActionListener(new StopHandler());
		killButton.addActionListener(new KillHandler());

		workerTable.getSelectionModel().addListSelectionListener(
			new WorkerSelectionHandler()
		);
	}

	public void windowActivated(WindowEvent e) {}
	public void windowClosed(WindowEvent e) {}
	public void windowClosing(WindowEvent e) {
		System.exit(0);
	}
	public void windowDeactivated(WindowEvent e) {}
	public void windowDeiconified(WindowEvent e) {}
	public void windowIconified(WindowEvent e) {}
	public void windowOpened(WindowEvent e) {}

	class ExitButtonHandler implements ActionListener {
		private Window w;

		public ExitButtonHandler(Window w) {
			this.w = w;
		}

		public void actionPerformed(ActionEvent e) {
			w.dispose();
			System.exit(0);
		}
	}

	class StatusChangeHandler implements StatusChangeListener {
		public void statusChanged(StatusChangeEvent sce) {
			int row = workerList.indexOf(sce.getSource());
			if(row < 0) return;

			wtm.fireTableRowsUpdated(row, row);
		}
	}

	class WorkerSelectionHandler implements ListSelectionListener {
		public void valueChanged(ListSelectionEvent e) {
			ListSelectionModel lsm = workerTable.getSelectionModel();

			boolean enableReconnect = false;
			boolean enableStop = false;
			boolean enableKill = false;
			
			if(!lsm.isSelectionEmpty()) {
				int min = lsm.getMinSelectionIndex();
				int max = lsm.getMaxSelectionIndex();
				//fix annoying bug where max is too big....
				max = (max >= workerList.size() ? workerList.size()-1 : max);

				for(int i = min; i <= max; i++) {
					enableReconnect |= lsm.isSelectedIndex(i)
						&& ((Worker)workerList.get(i)).getStatus()
							== Worker.DISCONNECTED;

					enableStop |= lsm.isSelectedIndex(i)
						&& ((Worker)workerList.get(i)).getStatus()
							== Worker.RUNNING;

					enableKill = true;
				}
			}
			reconnectButton.setEnabled(enableReconnect);
			stopButton.setEnabled(enableStop);
			killButton.setEnabled(enableKill);
		}
	}

	class ReconnectHandler implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			ListSelectionModel lsm = workerTable.getSelectionModel();

			int min = lsm.getMinSelectionIndex();
			int max = lsm.getMaxSelectionIndex();

			for(int i = min; i <= max; i++) {
				Worker w = (Worker)workerList.get(i);
				
				if(lsm.isSelectedIndex(i)
					&& w.getStatus() == Worker.DISCONNECTED)
				{
					w.reconnect();
				}
			}
		}
	}

	class StopHandler implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			ListSelectionModel lsm = workerTable.getSelectionModel();

			int min = lsm.getMinSelectionIndex();
			int max = lsm.getMaxSelectionIndex();

			for(int i = min; i <= max; i++) {
				Worker w = (Worker)workerList.get(i);
				
				if(lsm.isSelectedIndex(i)
					&& w.getStatus() == Worker.RUNNING)
				{
					w.abortTask();
				}
			}
		}
	}

	class KillHandler implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			ListSelectionModel lsm = workerTable.getSelectionModel();

			int min = lsm.getMinSelectionIndex();
			int max = lsm.getMaxSelectionIndex();

			for(int i = max; i >= min; i--) {
				Worker w = (Worker)workerList.get(i);
				
				if(lsm.isSelectedIndex(i)) {
					w.abortTask();
					w.interrupt();
					workerList.remove(i);
				}
			}

			wtm.fireTableRowsDeleted(min, max);
		}
	}
}
