import java.io.*;
import javax.swing.tree.*;

public class ConstantPoolInfo implements ElementInfo
{
	private int count;
	private PoolItem constants[];
	private StatusKeeper stats;

	public ConstantPoolInfo(DataInputStream is, StatusKeeper z) throws IOException{
		stats = z;
  		count = is.readUnsignedShort();
		constants = new PoolItem[ count ];

		stats.initProgressBar();
		stats.setMax(count);

		for ( int x = 1; x < count; x++)
		{
			constants[x] = PoolItem.pullPoolItem(this, is, stats, x);
			if ( !constants[x].resolved ) break;
			if ( constants[x].isDoubleWide() ) x++;
		}
	}

	public DefaultMutableTreeNode Describe() {
		//What do we do? We return a treenode with all the other
		DefaultMutableTreeNode result = new DefaultMutableTreeNode("Constant Pool: "+count+" items.");

		/* Notes:
			switch by tag.  cast to the appropriate class, call Describe() and tack that node onto our result.
		*/

		for ( int x = 1; x < count; x++)
		{
		  if(constants[x] != null){
			switch (constants[x].tag)
			{
				case PoolItem.CONSTANT_Class:			   result.add( ((Class_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Fieldref:			result.add( ((Fieldref_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Methodref:		   result.add( ((Methodref_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_String:			  result.add( ((String_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Integer:			 result.add( ((Integer_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Float:			   result.add( ((Float_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Long:				result.add( ((Long_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Double:			  result.add( ((Double_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_InterfaceMethodref:  result.add( ((IMethodref_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_NameAndType:		 result.add( ((NameAndType_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Utf8:				result.add( ((Utf8_info)constants[x]).Describe() ); break;
				case PoolItem.CONSTANT_Unicode:			 result.add( ((Unicode_info)constants[x]).Describe() ); break;
				default:									result.add( ((PoolItem)constants[x]).Describe() ); break;
			}
		  }
		}
		return result;
	}

	public String getUtf8(int index)
	{
		if ( index < constants.length )
			return constants[index].getUtf8();
		else
			return "";
	}

	public String describe(int c)
	{
		if ( c < constants.length )
			return constants[c].toString();
		else
			return "Invalid Index";
	}

	public String getName(int c)
	{
		if ( c < constants.length )
			return constants[c].getName();
		else
			return "InvalidIndex";
	}

	public String getSignature(int c)
	{
		if ( c < constants.length )
			return constants[c].getSignature();
		else
			return "InvalidIndex";
	}
}

class PoolItem implements ElementInfo
{
	public static final int CONSTANT_Class			   = 7;
	public static final int CONSTANT_Fieldref			= 9;
	public static final int CONSTANT_Methodref		  = 10;
	public static final int CONSTANT_String			  = 8;
	public static final int CONSTANT_Integer			 = 3;
	public static final int CONSTANT_Float			   = 4;
	public static final int CONSTANT_Long				= 5;
	public static final int CONSTANT_Double			  = 6;
	public static final int CONSTANT_InterfaceMethodref = 11;
	public static final int CONSTANT_NameAndType		= 12;
	public static final int CONSTANT_Utf8				= 1;
	public static final int CONSTANT_Unicode			 = 2;
	public static final int CONSTANT_Unknown			= -1;
	public static final int CONSTANT_Error			  = -2;

	protected int tag;
	protected ConstantPoolInfo constants;

	public boolean resolved;
	protected StatusKeeper stats;
	protected int num;
	public boolean basicType;

	protected DefaultMutableTreeNode Description;

	public PoolItem(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		resolved = false;
		basicType = false;
		tag = t;
		Description = new DefaultMutableTreeNode("PoolItemError");
		constants = c;
		System.out.println("Error: Unknown constant [" + t + "]");
		stats = x;
		num = y;
	}

	public PoolItem()
	{
		tag = CONSTANT_Unknown;
		resolved = false;
	}

	public boolean isDoubleWide()
	{
		if ( tag == CONSTANT_Double || tag == CONSTANT_Long ) return true;
		else return false;
	}

	public String getUtf8()
	{
		return "";
	}

	public String getName()
	{
		switch (tag)
		{
			case CONSTANT_Class:			   return "Class";
			case CONSTANT_Fieldref:			return "Fieldref";
			case CONSTANT_Methodref:		   return "Methodref";
			case CONSTANT_String:			  return "String";
			case CONSTANT_Integer:			 return "Integer";
			case CONSTANT_Float:			   return "Float";
			case CONSTANT_Long:				return "Long";
			case CONSTANT_Double:			  return "Double";
			case CONSTANT_InterfaceMethodref:  return "InterfaceMethodref";
			case CONSTANT_NameAndType:		 return "NameAndType";
			case CONSTANT_Utf8:				return "Utf8";
			case CONSTANT_Error:			   return "Error";
			case CONSTANT_Unknown:			 return "Unknown" + tag;
			default:						   return "Unknown " + tag;
		}
	}

	public String getSignature()
	{
		return "NONE";
	}

	public DefaultMutableTreeNode Describe()
	{
		return Description;//setTerminal(parent, getName()).toString();
	}

	public String toString()
	{
		return getName();
	}

	public static PoolItem pullPoolItem(ConstantPoolInfo c, DataInputStream is, StatusKeeper x, int y)
	{
		int t;
		try
		{
			t = is.readUnsignedByte();
		}
		catch ( IOException ioe)
		{
			System.out.println("poolitem" + ioe);
			t = CONSTANT_Error;
		}
		switch (t)
		{
			case CONSTANT_Class:			  return new Class_info	  (c, t, is, x, y);
			case CONSTANT_Fieldref:		   return new Fieldref_info   (c, t, is, x, y);
			case CONSTANT_Methodref:		  return new Methodref_info  (c, t, is, x, y);
			case CONSTANT_String:			 return new String_info	 (c, t, is, x, y);
			case CONSTANT_Integer:			return new Integer_info	(c, t, is, x, y);
			case CONSTANT_Float:			  return new Float_info	  (c, t, is, x, y);
			case CONSTANT_Long:			   return new Long_info	   (c, t, is, x, y);
			case CONSTANT_Double:			 return new Double_info	 (c, t, is, x, y);
			case CONSTANT_InterfaceMethodref: return new IMethodref_info (c, t, is, x, y);
			case CONSTANT_NameAndType:		return new NameAndType_info(c, t, is, x, y);
			case CONSTANT_Utf8:			   return new Utf8_info	   (c, t, is, x, y);
			case CONSTANT_Unicode:			return new Unicode_info	(c, t, is, x, y);
			default:						  return new PoolItem		(c, t, is, x, y);
		}
	}
}

class Utf8_info extends PoolItem
{
	int length;
	byte[] contents;
	char[] content;
	String s;

	public Utf8_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			tag = t;
			basicType = true;
			length = is.readUnsignedShort();
			contents = new byte[length];
			content = new char[length];
			is.readFully(contents);
			for(int i = 0; i < length; i++)
	  			content[i] = (char)((0 << 8) | (contents[i] & 0xff));

			String xx = new String(content);
			s = "";
			for(int i = 0; i < length; i++)
				if(xx.charAt(i) == '\n') s += "<crlf>";
				else s+= xx.charAt(i);
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			length = 0;
			s = "";
		}
		resolved = true;
	}

	public String getUtf8()
	{
		return s;
	}

	public String toString()
	{
		return "Utf8: [" + length + "] \"" + s + "\"";
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing UTF8 \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Unicode_info extends PoolItem
{
	int length;
	short[] contents;

	public Unicode_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			tag = t;
			basicType = true;
			length = is.readUnsignedShort();
			contents = new short[length];
			for ( int xx = 0; xx < length; xx++ )
				contents[xx] = is.readShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			length = 0;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Unicode: [" + length + "] " + contents;
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Unicode \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Class_info extends PoolItem
{
	int index;

	public Class_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Class: \"" + getName() +"\"";
	}

	public String getName()
	{
		return constants.getUtf8(index);
	}

	public String getUtf8()
	{
		return constants.getUtf8(index);
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Class \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Fieldref_info extends PoolItem
{
	int class_index, name_index;

	public Fieldref_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			class_index = is.readUnsignedShort();
			name_index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			class_index = -1;
			name_index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Field: " + constants.getName(class_index) + "." +
			constants.getName(name_index) + " " +
			constants.getSignature(name_index);
	}

	public String getName()
	{
		return constants.getName(name_index);
	}

	public String getSignature()
	{
		return constants.getSignature(name_index);
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Field \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Methodref_info extends PoolItem
{
	int class_index, name_index;

	public Methodref_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			class_index = is.readUnsignedShort();
			name_index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			class_index = -1;
			name_index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Method: " + constants.getName(class_index) + "." +
			constants.getName(name_index) +
			constants.getSignature(name_index);
	}

	public String getName()
	{
		return constants.getName(name_index);
	}

	public String getSignature()
	{
		return constants.getSignature(name_index);
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Method\""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class IMethodref_info extends PoolItem
{
	int class_index, name_index;

	public IMethodref_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			class_index = is.readUnsignedShort();
			name_index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			class_index = -1;
			name_index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "InterfaceMethod: " + constants.getName(class_index) + "." +
			constants.getName(name_index) +
			constants.getSignature(name_index);
	}

	public String getName()
	{
		return constants.getName(name_index);
	}

	public String getSignature()
	{
		return constants.getSignature(name_index);
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Interface Method Reference\""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class NameAndType_info extends PoolItem
{
	int signature_index, name_index;

	public NameAndType_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			basicType = true;
			name_index = is.readUnsignedShort();
			signature_index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			signature_index = -1;
			name_index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "NameAndType: " + constants.getUtf8(name_index) + " " +
			constants.getUtf8(signature_index);
	}

	public String getName()
	{
		return constants.getUtf8(name_index);
	}

	public String getSignature()
	{
		return constants.getUtf8(signature_index);
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Name & Type \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class String_info extends PoolItem
{
	int index;

	String_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			tag = t;
			constants = c;
			basicType = true;
			index = is.readUnsignedShort();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			index = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "String #" + index + " -> \"" +
			constants.getUtf8(index) + "\"";
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing String \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Integer_info extends PoolItem
{
	int value;

	Integer_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			tag = t;
			basicType = true;
			value = is.readInt();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			value = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Integer: " + value;
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Integer \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Float_info extends PoolItem
{
	float value;

	Float_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			basicType = true;
			tag = t;
			value = is.readFloat();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			value = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Float " + value;
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Float \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Long_info extends PoolItem
{
	long value;

	Long_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			basicType = true;
			tag = t;
			value = is.readLong();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			value = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Long " + value;
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Long \""+toString()+"\"...", num);
		return new DefaultMutableTreeNode(toString());
	}
}

class Double_info extends PoolItem
{
	double value;

	Double_info(ConstantPoolInfo c, int t, DataInputStream is, StatusKeeper x, int y)
	{
		try
		{
			stats = x;
			num = y;
			constants = c;
			basicType = true;
			tag = t;
			value = is.readDouble();
		}
		catch (IOException ioe)
		{
			System.out.println("" + ioe);
			value = -1;
		}
		resolved = true;
	}

	public String toString()
	{
		return "Double " + value;
	}

	public DefaultMutableTreeNode Describe()
	{
		stats.setProgressText("Processing Double \""+toString()+"\"...", num);   
		return new DefaultMutableTreeNode(toString());
	}
}
