/*
 * meminfo.c -- Memory statistics
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include "ugh.h"

static uint64_t mem_total;
static uint64_t mem_free;
static uint64_t buffers;
static uint64_t cached;
	
static uint64_t high_total;
static uint64_t high_free;
	
static uint64_t low_total;
static uint64_t low_free;

static uint64_t swap_total;
static uint64_t swap_free;

struct mem_table_struct {
	const char *tag;
	uint64_t *data;
};

static struct mem_table_struct mem_table[] = {
	{"MemTotal",	&mem_total},	
	{"MemFree",	&mem_free},
	{"Buffers",	&buffers},
	{"Cached",	&cached},
	{"HighTotal",	&high_total},
	{"HighFree",	&high_free},
	{"LowTotal",	&low_total},
	{"LowFree",	&low_free},
	{"SwapTotal",	&swap_total},
	{"SwapFree",	&swap_free},
};
const int mem_table_count = sizeof(mem_table)/sizeof(struct mem_table_struct);

/* Grab /proc/meminfo data */
static int grab_mem_info(void)
{
	FILE *fp;
	char buf[1024];
	int i, x;

	fp = fopen("/proc/meminfo", "r");
	if (!fp)
		return 0;

	while (fgets(buf, 1024, fp) != NULL) {
		for (i = 0; i < mem_table_count; i++) {
			x = strlen(mem_table[i].tag);
			if (!strncmp(mem_table[i].tag, buf, x)) {
				*mem_table[i].data = strtoll(buf + x + 1,
					NULL, 0);
				continue;
			}
		}
	}

	fclose(fp);

	return 1;
}

/* Process /proc/meminfo */
void process_meminfo(void)
{
	uint64_t mem_programs;
	int x, y, z, gchars, chars;
	
	if (!grab_mem_info())
		return;

	mem_programs = mem_total - (mem_free + buffers + cached);
	z = printf(" Mem: ");
	chars = get_window_width() - 1;;
	gchars = chars - z;
	
	y = my_round( ((double)mem_programs / mem_total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("P");
	
	y = my_round( ((double)buffers / mem_total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("B");
	
	y = my_round( ((double)cached / mem_total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("C");

	if (z < chars)
		for (x = chars - z; x > 0; x--)
			printf("-");
	printf("\n");

	if (swap_total) {
		z = printf("Swap: ");
		gchars = chars - z;
		y = my_round( ((double)(swap_total - swap_free) / swap_total) *
			gchars );
		for (x = 0; x < y && z < chars; x++, z++)
			printf("S");

		if (z < chars)
			for (x = chars - z; x > 0; x--)
				printf("-");
		printf("\n");
	}

	printf("     Programs: %"PRIu64"K   Buffers: %"PRIu64"K   Cache: %"PRIu64"K  Free: %"PRIu64"K" CLR_EOL "\n",
		mem_programs, buffers, cached, mem_free);

	 printf("     ");
	if (high_total)
		printf("Highmem Use: %.02f%%   ",
			100.0 - 100.0 * high_free / high_total);
	if (low_total)
		printf("Lowmem Use: %.02f%%   ",
			100.0 - 100.0 * low_free / low_total);
	if (swap_total)
		printf("Swap Use: %"PRIu64"K (%.02f%%)",
			swap_total - swap_free,
			100.0 - 100.0 * swap_free / swap_total);
	printf(CLR_EOL "\n");
}
